<?php

namespace App\Http\Controllers\Api;

use App\Models\Payment;
use App\Models\Invoice;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class PaymentController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Payment::where('tenant_id', $request->user()->tenant_id)
            ->with(['contact', 'invoice']);

        if ($request->filled('contact_id')) {
            $query->where('contact_id', $request->contact_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('payment_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('payment_date', '<=', $request->date_to);
        }

        $payments = $query->orderBy('payment_date', 'desc')
            ->paginate($request->get('per_page', 15));

        return $this->paginated($payments);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'contact_id' => 'required|exists:contacts,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'bank_account_id' => 'nullable|exists:bank_accounts,id',
            'payment_date' => 'required|date',
            'amount' => 'required|numeric|min:0.001',
            'currency' => 'nullable|string|size:3',
            'payment_method' => 'required|string|max:50',
            'reference' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:1000',
        ]);

        $payment = Payment::create([
            'tenant_id' => $request->user()->tenant_id,
            'contact_id' => $validated['contact_id'],
            'invoice_id' => $validated['invoice_id'] ?? null,
            'bank_account_id' => $validated['bank_account_id'] ?? null,
            'payment_date' => $validated['payment_date'],
            'amount' => $validated['amount'],
            'currency' => $validated['currency'] ?? 'BHD',
            'payment_method' => $validated['payment_method'],
            'reference' => $validated['reference'] ?? null,
            'description' => $validated['description'] ?? null,
            'status' => Payment::STATUS_COMPLETED,
            'created_by' => $request->user()->id,
        ]);

        if ($payment->invoice_id) {
            $payment->invoice->updatePaymentStatus();
        }

        return $this->success($payment->load(['contact', 'invoice']), 'Payment recorded successfully', 201);
    }

    public function show(Request $request, Payment $payment): JsonResponse
    {
        $this->authorize('view', $payment);
        return $this->success($payment->load(['contact', 'invoice', 'bankAccount', 'createdBy']));
    }

    public function destroy(Request $request, Payment $payment): JsonResponse
    {
        $this->authorize('delete', $payment);

        if ($payment->status === Payment::STATUS_REFUNDED) {
            return $this->error('Cannot delete a refunded payment', 422);
        }

        $invoice = $payment->invoice;
        $payment->delete();

        if ($invoice) {
            $invoice->updatePaymentStatus();
        }

        return $this->success(null, 'Payment deleted successfully');
    }

    public function refund(Request $request, Payment $payment): JsonResponse
    {
        $this->authorize('update', $payment);

        $validated = $request->validate([
            'amount' => 'nullable|numeric|min:0.001|max:' . ($payment->amount - ($payment->refunded_amount ?? 0)),
            'reason' => 'nullable|string|max:500',
        ]);

        $refundAmount = $validated['amount'] ?? $payment->amount;

        $payment->update([
            'refunded_amount' => ($payment->refunded_amount ?? 0) + $refundAmount,
            'refund_reason' => $validated['reason'] ?? null,
            'status' => $refundAmount >= $payment->amount ? Payment::STATUS_REFUNDED : Payment::STATUS_PARTIALLY_REFUNDED,
        ]);

        if ($payment->invoice) {
            $payment->invoice->updatePaymentStatus();
        }

        return $this->success($payment->fresh(), 'Payment refunded successfully');
    }

    public function summary(Request $request): JsonResponse
    {
        $tenantId = $request->user()->tenant_id;
        $today = now();

        return $this->success([
            'this_month' => Payment::where('tenant_id', $tenantId)
                ->whereMonth('payment_date', $today->month)
                ->whereYear('payment_date', $today->year)
                ->where('status', Payment::STATUS_COMPLETED)
                ->sum('amount'),
            'last_month' => Payment::where('tenant_id', $tenantId)
                ->whereMonth('payment_date', $today->subMonth()->month)
                ->whereYear('payment_date', $today->year)
                ->where('status', Payment::STATUS_COMPLETED)
                ->sum('amount'),
            'by_method' => Payment::where('tenant_id', $tenantId)
                ->whereMonth('payment_date', now()->month)
                ->where('status', Payment::STATUS_COMPLETED)
                ->selectRaw('payment_method, SUM(amount) as total')
                ->groupBy('payment_method')
                ->pluck('total', 'payment_method'),
        ]);
    }
}
