<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreLeaveRequest;
use App\Models\Employee;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use App\Services\LeaveService;
use Illuminate\Http\Request;

class LeaveRequestController extends Controller
{
    public function __construct(protected LeaveService $service) {}

    public function index(Request $request)
    {
        $query = LeaveRequest::where('tenant_id', $request->user()->tenant_id)
            ->with(['employee.user', 'leaveType']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->employee_id) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->upcoming) {
            $query->where('start_date', '>=', now()->toDateString())
                ->where('status', 'approved');
        }

        $leaves = $query->orderByDesc('created_at')->paginate($request->per_page ?? 20);

        return response()->json([
            'data' => $leaves->items(),
            'meta' => [
                'total' => $leaves->total(),
                'per_page' => $leaves->perPage(),
                'current_page' => $leaves->currentPage(),
            ],
        ]);
    }

    public function store(StoreLeaveRequest $request)
    {
        $employee = Employee::where('user_id', $request->user()->id)->firstOrFail();

        $leaveRequest = $this->service->submit(
            $request->user()->tenant_id,
            $employee->id,
            $request->validated()
        );

        return response()->json(['data' => $leaveRequest->load(['leaveType', 'employee.user'])], 201);
    }

    public function show(LeaveRequest $leaveRequest)
    {
        $this->authorize('view', $leaveRequest);
        return response()->json(['data' => $leaveRequest->load(['employee.user', 'leaveType', 'approvedBy'])]);
    }

    public function approve(Request $request, LeaveRequest $leaveRequest)
    {
        $this->authorize('approve', $leaveRequest);

        $this->service->approve($leaveRequest, $request->user()->id);

        return response()->json(['data' => $leaveRequest->fresh()->load(['employee.user', 'leaveType'])]);
    }

    public function reject(Request $request, LeaveRequest $leaveRequest)
    {
        $this->authorize('approve', $leaveRequest);

        $validated = $request->validate(['reason' => 'required|string|max:500']);

        $this->service->reject($leaveRequest, $request->user()->id, $validated['reason']);

        return response()->json(['data' => $leaveRequest->fresh()->load(['employee.user', 'leaveType'])]);
    }

    public function cancel(LeaveRequest $leaveRequest)
    {
        $this->authorize('update', $leaveRequest);

        if ($leaveRequest->status !== 'pending') {
            return response()->json(['message' => 'Can only cancel pending requests'], 422);
        }

        $leaveRequest->update(['status' => 'cancelled']);

        return response()->json(['data' => $leaveRequest]);
    }

    public function balances(Request $request)
    {
        $employee = Employee::where('user_id', $request->user()->id)->firstOrFail();

        $balances = LeaveBalance::where('employee_id', $employee->id)
            ->where('year', now()->year)
            ->with('leaveType')
            ->get()
            ->map(fn($b) => [
                'leave_type_id' => $b->leave_type_id,
                'leave_type' => $b->leaveType->name,
                'entitled' => $b->entitled,
                'used' => $b->used,
                'remaining' => $b->remaining,
            ]);

        return response()->json(['data' => $balances]);
    }
}
