<?php

namespace App\Http\Controllers\Api;

use App\Models\Invoice;
use App\Models\Bill;
use App\Models\Opportunity;
use App\Models\Task;
use App\Models\Contact;
use App\Services\ReportService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class DashboardController extends ApiController
{
    public function __construct(protected ReportService $reportService) {}

    public function index(Request $request): JsonResponse
    {
        $tenantId = $request->user()->tenant_id;
        $startOfMonth = Carbon::now()->startOfMonth();
        $endOfMonth = Carbon::now()->endOfMonth();

        return $this->success([
            'revenue' => $this->getRevenueStats($tenantId, $startOfMonth, $endOfMonth),
            'invoices' => $this->getInvoiceStats($tenantId),
            'bills' => $this->getBillStats($tenantId),
            'pipeline' => $this->getPipelineStats($tenantId),
            'tasks' => $this->getTaskStats($tenantId, $request->user()->id),
            'recent_activities' => $this->getRecentActivities($tenantId),
        ]);
    }

    protected function getRevenueStats(int $tenantId, Carbon $start, Carbon $end): array
    {
        $currentMonth = Invoice::where('tenant_id', $tenantId)
            ->whereBetween('invoice_date', [$start, $end])
            ->whereIn('status', ['sent', 'paid', 'partial'])
            ->sum('total');

        $lastMonth = Invoice::where('tenant_id', $tenantId)
            ->whereBetween('invoice_date', [$start->copy()->subMonth(), $end->copy()->subMonth()])
            ->whereIn('status', ['sent', 'paid', 'partial'])
            ->sum('total');

        $change = $lastMonth > 0 ? (($currentMonth - $lastMonth) / $lastMonth) * 100 : 0;

        return [
            'current' => round($currentMonth, 3),
            'previous' => round($lastMonth, 3),
            'change' => round($change, 1),
            'currency' => 'BHD',
        ];
    }

    protected function getInvoiceStats(int $tenantId): array
    {
        return [
            'outstanding' => Invoice::where('tenant_id', $tenantId)
                ->whereIn('status', ['sent', 'partial'])
                ->selectRaw('SUM(total - amount_paid) as total')
                ->value('total') ?? 0,
            'overdue' => Invoice::where('tenant_id', $tenantId)
                ->whereIn('status', ['sent', 'partial'])
                ->where('due_date', '<', now())
                ->selectRaw('SUM(total - amount_paid) as total')
                ->value('total') ?? 0,
            'draft_count' => Invoice::where('tenant_id', $tenantId)
                ->where('status', 'draft')
                ->count(),
        ];
    }

    protected function getBillStats(int $tenantId): array
    {
        return [
            'outstanding' => Bill::where('tenant_id', $tenantId)
                ->whereIn('status', ['received', 'partial'])
                ->selectRaw('SUM(total - amount_paid) as total')
                ->value('total') ?? 0,
            'overdue' => Bill::where('tenant_id', $tenantId)
                ->whereIn('status', ['received', 'partial'])
                ->where('due_date', '<', now())
                ->selectRaw('SUM(total - amount_paid) as total')
                ->value('total') ?? 0,
            'due_this_week' => Bill::where('tenant_id', $tenantId)
                ->whereIn('status', ['received', 'partial'])
                ->whereBetween('due_date', [now(), now()->addWeek()])
                ->selectRaw('SUM(total - amount_paid) as total')
                ->value('total') ?? 0,
        ];
    }

    protected function getPipelineStats(int $tenantId): array
    {
        $opportunities = Opportunity::where('tenant_id', $tenantId)
            ->where('status', 'open');

        return [
            'total_value' => $opportunities->sum('value'),
            'weighted_value' => $opportunities->selectRaw('SUM(value * probability / 100) as total')->value('total') ?? 0,
            'count' => $opportunities->count(),
            'won_this_month' => Opportunity::where('tenant_id', $tenantId)
                ->where('status', 'won')
                ->whereMonth('closed_at', now()->month)
                ->sum('value'),
        ];
    }

    protected function getTaskStats(int $tenantId, int $userId): array
    {
        $myTasks = Task::where('tenant_id', $tenantId)
            ->whereHas('assignees', fn($q) => $q->where('user_id', $userId))
            ->whereHas('status', fn($q) => $q->where('is_completed', false));

        return [
            'my_tasks' => $myTasks->count(),
            'overdue' => $myTasks->clone()->where('due_date', '<', now())->count(),
            'due_today' => $myTasks->clone()->whereDate('due_date', now())->count(),
            'due_this_week' => $myTasks->clone()->whereBetween('due_date', [now(), now()->addWeek()])->count(),
        ];
    }

    protected function getRecentActivities(int $tenantId): array
    {
        return \App\Models\Activity::where('tenant_id', $tenantId)
            ->with(['user:id,first_name,last_name', 'contact:id,first_name,last_name'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->toArray();
    }

    public function widget(Request $request, string $widget): JsonResponse
    {
        $tenantId = $request->user()->tenant_id;
        $period = $request->get('period', 'month');

        $data = match ($widget) {
            'revenue_chart' => $this->getRevenueChart($tenantId, $period),
            'pipeline_chart' => $this->reportService->getSalesPipeline($tenantId),
            'top_customers' => $this->getTopCustomers($tenantId),
            'recent_invoices' => $this->getRecentInvoices($tenantId),
            'upcoming_tasks' => $this->getUpcomingTasks($tenantId, $request->user()->id),
            default => [],
        };

        return $this->success($data);
    }

    protected function getRevenueChart(int $tenantId, string $period): array
    {
        $months = $period === 'year' ? 12 : 6;
        $data = [];

        for ($i = $months - 1; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $revenue = Invoice::where('tenant_id', $tenantId)
                ->whereYear('invoice_date', $date->year)
                ->whereMonth('invoice_date', $date->month)
                ->whereIn('status', ['sent', 'paid', 'partial'])
                ->sum('total');

            $data[] = [
                'month' => $date->format('M Y'),
                'revenue' => round($revenue, 3),
            ];
        }

        return $data;
    }

    protected function getTopCustomers(int $tenantId): array
    {
        return Contact::where('tenant_id', $tenantId)
            ->withSum(['invoices' => fn($q) => $q->whereIn('status', ['sent', 'paid', 'partial'])], 'total')
            ->orderByDesc('invoices_sum_total')
            ->limit(5)
            ->get(['id', 'first_name', 'last_name', 'company_name'])
            ->map(fn($c) => [
                'name' => $c->company_name ?: $c->full_name,
                'total' => round($c->invoices_sum_total ?? 0, 3),
            ])
            ->toArray();
    }

    protected function getRecentInvoices(int $tenantId): array
    {
        return Invoice::where('tenant_id', $tenantId)
            ->with('contact:id,first_name,last_name,company_name')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get(['id', 'invoice_number', 'contact_id', 'total', 'status', 'due_date'])
            ->toArray();
    }

    protected function getUpcomingTasks(int $tenantId, int $userId): array
    {
        return Task::where('tenant_id', $tenantId)
            ->whereHas('assignees', fn($q) => $q->where('user_id', $userId))
            ->whereHas('status', fn($q) => $q->where('is_completed', false))
            ->where('due_date', '>=', now())
            ->orderBy('due_date')
            ->limit(5)
            ->get(['id', 'title', 'due_date', 'priority'])
            ->toArray();
    }
}
