<?php

namespace App\Http\Controllers\Api;

use App\Models\Contact;
use App\Http\Requests\ContactRequest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ContactController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Contact::query()
            ->with(['owner', 'tags', 'company'])
            ->when($request->search, fn($q, $search) => 
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('company_name', 'like', "%{$search}%")
            )
            ->when($request->status, fn($q, $status) => $q->where('status', $status))
            ->when($request->lead_status, fn($q, $status) => $q->where('lead_status', $status))
            ->when($request->owner_id, fn($q, $owner) => $q->where('owner_id', $owner))
            ->when($request->tag_id, fn($q, $tag) => 
                $q->whereHas('tags', fn($t) => $t->where('tags.id', $tag))
            )
            ->when($request->type, fn($q, $type) => $q->where('type', $type))
            ->orderBy($request->sort_by ?? 'created_at', $request->sort_dir ?? 'desc');

        $contacts = $query->paginate($request->per_page ?? 25);

        return $this->paginated($contacts);
    }

    public function store(ContactRequest $request): JsonResponse
    {
        $contact = Contact::create($request->validated());

        if ($request->tags) {
            $contact->tags()->sync($request->tags);
        }

        return $this->success($contact->load(['owner', 'tags']), 'Contact created successfully', 201);
    }

    public function show(Contact $contact): JsonResponse
    {
        return $this->success(
            $contact->load(['owner', 'tags', 'company', 'activities', 'opportunities', 'invoices'])
        );
    }

    public function update(ContactRequest $request, Contact $contact): JsonResponse
    {
        $contact->update($request->validated());

        if ($request->has('tags')) {
            $contact->tags()->sync($request->tags);
        }

        return $this->success($contact->load(['owner', 'tags']), 'Contact updated successfully');
    }

    public function destroy(Contact $contact): JsonResponse
    {
        $contact->delete();
        return $this->success(null, 'Contact deleted successfully');
    }

    public function import(Request $request): JsonResponse
    {
        $request->validate([
            'file' => 'required|file|mimes:csv,xlsx,xls|max:10240',
        ]);

        // Import logic here
        return $this->success(null, 'Import started');
    }

    public function export(Request $request): JsonResponse
    {
        // Export logic here
        return $this->success(['url' => '/exports/contacts.csv'], 'Export ready');
    }

    public function merge(Request $request): JsonResponse
    {
        $request->validate([
            'primary_id' => 'required|exists:contacts,id',
            'secondary_ids' => 'required|array',
            'secondary_ids.*' => 'exists:contacts,id',
        ]);

        $primary = Contact::find($request->primary_id);
        
        foreach ($request->secondary_ids as $id) {
            $secondary = Contact::find($id);
            // Merge logic - move related records, then delete secondary
            $secondary->opportunities()->update(['contact_id' => $primary->id]);
            $secondary->activities()->update(['activitable_id' => $primary->id]);
            $secondary->delete();
        }

        return $this->success($primary->fresh(), 'Contacts merged successfully');
    }

    public function timeline(Contact $contact): JsonResponse
    {
        $activities = $contact->activities()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return $this->paginated($activities);
    }
}
