<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Validation\ValidationException;

class AuthController extends ApiController
{
    public function login(Request $request): JsonResponse
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        if ($user->status !== 'active') {
            return $this->error('Your account is not active', 403);
        }

        $user->update(['last_login_at' => now()]);

        $token = $user->createToken('auth-token')->plainTextToken;

        return $this->success([
            'user' => $user->load('tenant'),
            'token' => $token,
        ], 'Login successful');
    }

    public function register(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'company_name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'country' => 'nullable|string|size:2',
        ]);

        // Create tenant
        $tenant = Tenant::create([
            'name' => $request->company_name,
            'slug' => \Str::slug($request->company_name) . '-' . \Str::random(6),
            'email' => $request->email,
            'phone' => $request->phone,
            'country' => $request->country ?? 'BH',
            'currency' => 'BHD',
            'timezone' => 'Asia/Bahrain',
            'status' => 'active',
            'plan' => 'trial',
            'trial_ends_at' => now()->addDays(14),
            'vat_rate' => 10.00,
        ]);

        // Create user
        $user = User::create([
            'tenant_id' => $tenant->id,
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,
            'is_super_admin' => true,
            'status' => 'active',
        ]);

        $token = $user->createToken('auth-token')->plainTextToken;

        return $this->success([
            'user' => $user->load('tenant'),
            'token' => $token,
        ], 'Registration successful', 201);
    }

    public function user(Request $request): JsonResponse
    {
        return $this->success($request->user()->load(['tenant', 'teams']));
    }

    public function logout(Request $request): JsonResponse
    {
        $request->user()->currentAccessToken()->delete();
        return $this->success(null, 'Logged out successfully');
    }

    public function updateProfile(Request $request): JsonResponse
    {
        $user = $request->user();

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'avatar' => 'nullable|image|max:2048',
            'job_title' => 'nullable|string|max:100',
            'bio' => 'nullable|string|max:500',
            'timezone' => 'nullable|string|max:50',
            'language' => 'nullable|string|size:2',
        ]);

        if ($request->hasFile('avatar')) {
            $validated['avatar'] = $request->file('avatar')->store('avatars', 'public');
        }

        $user->update($validated);

        return $this->success($user->fresh(), 'Profile updated');
    }

    public function updatePassword(Request $request): JsonResponse
    {
        $request->validate([
            'current_password' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user = $request->user();

        if (!Hash::check($request->current_password, $user->password)) {
            return $this->error('Current password is incorrect', 422);
        }

        $user->update(['password' => Hash::make($request->password)]);

        return $this->success(null, 'Password updated');
    }

    public function forgotPassword(Request $request): JsonResponse
    {
        $request->validate(['email' => 'required|email']);

        $status = Password::sendResetLink($request->only('email'));

        return $status === Password::RESET_LINK_SENT
            ? $this->success(null, 'Password reset link sent')
            : $this->error('Unable to send reset link', 422);
    }

    public function resetPassword(Request $request): JsonResponse
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->update(['password' => Hash::make($password)]);
            }
        );

        return $status === Password::PASSWORD_RESET
            ? $this->success(null, 'Password reset successful')
            : $this->error('Unable to reset password', 422);
    }
}

class DashboardController extends ApiController
{
    public function index(): JsonResponse
    {
        $tenantId = auth()->user()->tenant_id;

        return $this->success([
            'crm' => $this->crmMetrics($tenantId),
            'accounting' => $this->accountingMetrics($tenantId),
            'projects' => $this->projectMetrics($tenantId),
            'recent_activities' => $this->recentActivities($tenantId),
        ]);
    }

    public function crm(): JsonResponse
    {
        $tenantId = auth()->user()->tenant_id;

        $opportunities = \App\Models\Opportunity::where('tenant_id', $tenantId);
        $contacts = \App\Models\Contact::where('tenant_id', $tenantId);

        return $this->success([
            'total_opportunities' => $opportunities->count(),
            'open_opportunities' => $opportunities->clone()->where('status', 'open')->count(),
            'won_this_month' => $opportunities->clone()->where('status', 'won')->whereMonth('actual_close_date', now()->month)->count(),
            'pipeline_value' => $opportunities->clone()->where('status', 'open')->sum('value'),
            'won_value_this_month' => $opportunities->clone()->where('status', 'won')->whereMonth('actual_close_date', now()->month)->sum('value'),
            'total_contacts' => $contacts->count(),
            'new_contacts_this_month' => $contacts->clone()->whereMonth('created_at', now()->month)->count(),
            'opportunities_by_stage' => $opportunities->clone()
                ->selectRaw('stage_id, COUNT(*) as count, SUM(value) as value')
                ->with('stage:id,name,color')
                ->groupBy('stage_id')
                ->get(),
            'conversion_rate' => $this->calculateConversionRate($tenantId),
        ]);
    }

    public function accounting(): JsonResponse
    {
        $tenantId = auth()->user()->tenant_id;
        $thisMonth = now()->startOfMonth();
        $lastMonth = now()->subMonth()->startOfMonth();

        $invoices = \App\Models\Invoice::where('tenant_id', $tenantId);
        $payments = \App\Models\Payment::where('tenant_id', $tenantId);
        $expenses = \App\Models\Expense::where('tenant_id', $tenantId);

        $revenueThisMonth = $payments->clone()->whereMonth('payment_date', now()->month)->sum('amount');
        $revenueLastMonth = $payments->clone()->whereMonth('payment_date', now()->subMonth()->month)->sum('amount');
        $expensesThisMonth = $expenses->clone()->whereMonth('expense_date', now()->month)->sum('total');

        return $this->success([
            'revenue_this_month' => $revenueThisMonth,
            'revenue_last_month' => $revenueLastMonth,
            'revenue_growth' => $revenueLastMonth > 0 ? round((($revenueThisMonth - $revenueLastMonth) / $revenueLastMonth) * 100, 1) : 0,
            'expenses_this_month' => $expensesThisMonth,
            'profit_this_month' => $revenueThisMonth - $expensesThisMonth,
            'outstanding_invoices' => $invoices->clone()->whereIn('status', ['sent', 'viewed', 'partial', 'overdue'])->sum('amount_due'),
            'overdue_invoices' => $invoices->clone()->where('status', 'overdue')->sum('amount_due'),
            'overdue_count' => $invoices->clone()->where('status', 'overdue')->count(),
            'invoices_by_status' => $invoices->clone()
                ->selectRaw('status, COUNT(*) as count, SUM(total) as total')
                ->groupBy('status')
                ->get(),
            'revenue_by_month' => $this->revenueByMonth($tenantId),
        ]);
    }

    public function projects(): JsonResponse
    {
        $tenantId = auth()->user()->tenant_id;

        $projects = \App\Models\Project::where('tenant_id', $tenantId);
        $tasks = \App\Models\Task::where('tenant_id', $tenantId);

        return $this->success([
            'total_projects' => $projects->count(),
            'active_projects' => $projects->clone()->where('status', 'active')->count(),
            'completed_projects' => $projects->clone()->where('status', 'completed')->count(),
            'total_tasks' => $tasks->count(),
            'completed_tasks' => $tasks->clone()->whereNotNull('completed_at')->count(),
            'overdue_tasks' => $tasks->clone()->whereNull('completed_at')->whereDate('due_date', '<', now())->count(),
            'tasks_due_today' => $tasks->clone()->whereNull('completed_at')->whereDate('due_date', today())->count(),
            'my_tasks' => $tasks->clone()->where('assignee_id', auth()->id())->whereNull('completed_at')->count(),
            'projects_by_status' => $projects->clone()
                ->selectRaw('status, COUNT(*) as count')
                ->groupBy('status')
                ->get(),
            'hours_logged_this_week' => \App\Models\TimeEntry::where('tenant_id', $tenantId)
                ->whereBetween('date', [now()->startOfWeek(), now()->endOfWeek()])
                ->sum('hours'),
        ]);
    }

    protected function crmMetrics(int $tenantId): array
    {
        return [
            'open_opportunities' => \App\Models\Opportunity::where('tenant_id', $tenantId)->where('status', 'open')->count(),
            'pipeline_value' => \App\Models\Opportunity::where('tenant_id', $tenantId)->where('status', 'open')->sum('value'),
            'new_contacts_today' => \App\Models\Contact::where('tenant_id', $tenantId)->whereDate('created_at', today())->count(),
        ];
    }

    protected function accountingMetrics(int $tenantId): array
    {
        return [
            'revenue_this_month' => \App\Models\Payment::where('tenant_id', $tenantId)->whereMonth('payment_date', now()->month)->sum('amount'),
            'outstanding_invoices' => \App\Models\Invoice::where('tenant_id', $tenantId)->whereIn('status', ['sent', 'viewed', 'partial', 'overdue'])->sum('amount_due'),
            'overdue_count' => \App\Models\Invoice::where('tenant_id', $tenantId)->where('status', 'overdue')->count(),
        ];
    }

    protected function projectMetrics(int $tenantId): array
    {
        return [
            'active_projects' => \App\Models\Project::where('tenant_id', $tenantId)->where('status', 'active')->count(),
            'my_tasks_pending' => \App\Models\Task::where('tenant_id', $tenantId)->where('assignee_id', auth()->id())->whereNull('completed_at')->count(),
            'tasks_due_today' => \App\Models\Task::where('tenant_id', $tenantId)->whereNull('completed_at')->whereDate('due_date', today())->count(),
        ];
    }

    protected function recentActivities(int $tenantId): \Illuminate\Database\Eloquent\Collection
    {
        return \App\Models\Activity::where('tenant_id', $tenantId)
            ->with(['user', 'activitable'])
            ->latest()
            ->limit(10)
            ->get();
    }

    protected function calculateConversionRate(int $tenantId): float
    {
        $total = \App\Models\Opportunity::where('tenant_id', $tenantId)->whereMonth('created_at', now()->month)->count();
        $won = \App\Models\Opportunity::where('tenant_id', $tenantId)->where('status', 'won')->whereMonth('actual_close_date', now()->month)->count();

        return $total > 0 ? round(($won / $total) * 100, 1) : 0;
    }

    protected function revenueByMonth(int $tenantId): array
    {
        return \App\Models\Payment::where('tenant_id', $tenantId)
            ->whereYear('payment_date', now()->year)
            ->selectRaw('MONTH(payment_date) as month, SUM(amount) as total')
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->pluck('total', 'month')
            ->toArray();
    }
}
