<?php

namespace App\Http\Controllers\Api;

use App\Models\Activity;
use App\Models\Contact;
use Illuminate\Http\Request;

class ActivityController extends ApiController
{
    public function index(Request $request)
    {
        $query = Activity::where('tenant_id', $request->user()->tenant_id)
            ->with(['contact', 'user']);

        if ($request->type) {
            $query->where('type', $request->type);
        }

        if ($request->contact_id) {
            $query->where('contact_id', $request->contact_id);
        }

        if ($request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->has('is_completed')) {
            $query->where('is_completed', $request->boolean('is_completed'));
        }

        if ($request->from_date) {
            $query->whereDate('scheduled_at', '>=', $request->from_date);
        }

        if ($request->to_date) {
            $query->whereDate('scheduled_at', '<=', $request->to_date);
        }

        $activities = $query->orderBy('scheduled_at', 'desc')
            ->paginate($request->per_page ?? 20);

        return $this->paginated($activities);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'contact_id' => 'required|exists:contacts,id',
            'type' => 'required|in:call,email,meeting,note,task,sms,whatsapp,follow_up',
            'subject' => 'required|string|max:255',
            'description' => 'nullable|string',
            'scheduled_at' => 'nullable|date',
            'due_at' => 'nullable|date',
            'duration_minutes' => 'nullable|integer|min:1',
        ]);

        $activity = Activity::create([
            'tenant_id' => $request->user()->tenant_id,
            'user_id' => $request->user()->id,
            ...$validated,
        ]);

        return $this->success($activity->load('contact', 'user'), 'Activity created successfully', 201);
    }

    public function show(Activity $activity)
    {
        return $this->success($activity->load('contact', 'user'));
    }

    public function update(Request $request, Activity $activity)
    {
        $validated = $request->validate([
            'subject' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'scheduled_at' => 'nullable|date',
            'due_at' => 'nullable|date',
            'duration_minutes' => 'nullable|integer|min:1',
            'outcome' => 'nullable|string|max:50',
        ]);

        $activity->update($validated);

        return $this->success($activity, 'Activity updated successfully');
    }

    public function destroy(Activity $activity)
    {
        $activity->delete();
        return $this->success(null, 'Activity deleted successfully');
    }

    public function complete(Request $request, Activity $activity)
    {
        $validated = $request->validate([
            'outcome' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
        ]);

        $activity->complete($validated['outcome'] ?? null);

        if (!empty($validated['notes'])) {
            $activity->update(['description' => $activity->description . "\n\nNotes: " . $validated['notes']]);
        }

        return $this->success($activity, 'Activity completed');
    }

    public function upcoming(Request $request)
    {
        $activities = Activity::where('tenant_id', $request->user()->tenant_id)
            ->where('user_id', $request->user()->id)
            ->upcoming()
            ->with('contact')
            ->take(10)
            ->get();

        return $this->success($activities);
    }

    public function overdue(Request $request)
    {
        $activities = Activity::where('tenant_id', $request->user()->tenant_id)
            ->where('user_id', $request->user()->id)
            ->overdue()
            ->with('contact')
            ->get();

        return $this->success($activities);
    }
}
